#!/bin/bash
#
# Usage: install_daemon.sh <imagefile>
#
# VERSION=24
# CHANGES="checks for new Hardware, and if existing disallows downgrade to FW < 16.11"

# constants #
BEROCONF=/usr/fallback/beroconf
PIDFILE=/var/run/install_daemon.pid

file=${1}
image_dir=/tmp/images

# functions #
function log {

	prefix=${1:common}
	logfile=/usr/conf/permlog/install-${prefix}.log
	pipe=/tmp/install_pipe

	if [ -p ${pipe} ]; then
		echo "[install_${prefix}] $(date): ${2}" &>${pipe}
	fi

	echo "[install_${prefix}] $(date): ${2}" >> ${logfile}
}

function fix_old_image_dir {
	mount -o remount,rw /
	if [ -d /mnt/images ]; then
		rm -rf /mnt/images
	fi
	ln -s /tmp /mnt/images
	mount -o remount,ro /
}

function tmp_cleanup {
	for i in "VERSION content.txt compare-content.txt"; do
		rm -f ${i}
	done
}

# main #
if [ ! -d ${image_dir} ]; then
	mkdir -p ${image_dir}
fi

if [ -z "${file}" ] ; then
	echo "ERROR: No filename given, leaving."
	rm -f ${PIDFILE}
	exit 1
fi

if [ ! -f "${image_dir}/${file}" ] ; then
	echo "ERROR: File ${image_dir}/${file} not found, leaving."
	rm -f ${PIDFILE}
	exit 1
fi

if [ ! -L /mnt/images ]; then
	fix_old_image_dir
fi

cd /tmp && tar zxf ${image_dir}/${file} VERSION
. /tmp/VERSION

# check if TYPE-field is sane.
case "${TYPE}" in
	fpga|firmware|kernel|rootfs|appfs|userappfs)
		;;
	*)
		log "ERROR: This package does not contain a valid type, leaving."
		rm -f ${PIDFILE}
		exit 1
		;;
esac

# remove previous log-file
rm -f /usr/conf/permlog/install-${TYPE}.log

log "${TYPE}" "Installing '${file}'."
case "${TYPE}" in
	fpga)
		log "${TYPE}" "Unpacking FPGA-Image."
		cd /tmp && tar zxf ${image_dir}/${file} fpga.img
		if [ ! -f /tmp/fpga.img ] ; then
			log "${TYPE}" "ERROR: Could not unpack the FPGA-Image, leaving."
			exit 1
		fi

		# load the kernel-module
		/sbin/insmod /usr/local/modules/bfpga.ko
		if [ ${?} -ne 0 ]; then
			log "${TYPE}" "ERROR: Could not load the Kernel-Module needed to program the FPGA, leaving."
			exit 1
		fi

		# retrieve Version-Information
		VERSION_INSTALLED="unknown"
		if [ -f /usr/conf/VERSION.fpga ]; then
			VERSION_INSTALLED="$(expr match "$(cat /usr/conf/VERSION.fpga)" ".*FPGA=\(.*\)")"
		fi
		VERSION_TOINSTALL="$(expr match "$(cat /tmp/VERSION)" ".*FPGA=\(.*\)")"

		log "${TYPE}" "Installing FPGA-Image. This process may take 3 minutes."
		log "${TYPE}" "Do NOT turn off your beroNet VoIP Gateway!"
		log "${TYPE}" "Updating FPGA-Image from Version ${VERSION_INSTALLED} to ${VERSION_TOINSTALL}."

		# prepare the next reboot
		${BEROCONF} set root boot_fwupdate 0
		echo "1" > /tmp/reboot_after_update

		# log update-operations
		echo "[${TYPE}] $(date): ${VERSION_INSTALLED} => ${VERSION_TOINSTALL}" >> /usr/conf/update.log
		mv /tmp/VERSION /usr/conf/VERSION.fpga

		# Perform the Update
		sleep 2
		dd if=/tmp/fpga.img of=/dev/bfpga bs=512k count=1
		;;
	firmware)
		log "${TYPE}" "Unpacking Firmware."
		cd /tmp && tar zxf ${image_dir}/${file} firmware.axf
		if [ ! -f /tmp/firmware.axf ]; then
			log "${TYPE}" "ERROR: Could not unpack the Firmware-Image, leaving."
			exit 1
		fi

		# retrieve Version-Information
		VERSION_INSTALLED="unknown"
		if [ -f /usr/conf/VERSION.firmware ]; then
			VERSION_INSTALLED="$(expr match "$(cat /usr/conf/VERSION.firmware)" ".*FIRMWARE=\(.*\)")"
		fi
		VERSION_TOINSTALL="$(expr match "$(cat /tmp/VERSION)" ".*FIRMWARE=\(.*\)")"

		log "${TYPE}" "Installing Firmware-Image. This process may take 3 minutes."
		log "${TYPE}" "Do NOT turn off your beroNet VoIP Gateway!"
		log "${TYPE}" "Updating Firmware-Image from Version ${VERSION_INSTALLED} to ${VERSION_TOINSTALL}."

		# prepare the next reboot
		${BEROCONF} set root boot_fwupdate 0
		echo "1" > /tmp/reboot_after_update

		# log update-operations
		echo "[${TYPE}] $(date): ${VERSION_INSTALLED} => ${VERSION_TOINSTALL}" >> /usr/conf/update.log
		mv /tmp/VERSION /usr/conf/VERSION.firmware

		# Perform the Update
		sleep 2
		dd if=/tmp/firmware.axf of=/dev/mtdblock3
	;;
	kernel)
		log "${TYPE}" "Unpacking Kernel-Image."
		cd /tmp && tar zxf ${image_dir}/${file} zImage
		if [ ! -f /tmp/zImage ]; then
			log "${TYPE}" "ERROR: Could not unpack the Kernel-Image, leaving."
			exit 1
		fi

		# Retrieve Version-Information
		VERSION_INSTALLED="unknown"
		if [ -f /usr/conf/VERSION.${TYPE} ]; then
			VERSION_INSTALLED="$($(expr match "$(cat /usr/conf/VERSION.kernel)" ".*KERNEL=\(.*\)"))"
		fi
		VERSION_TOINSTALL="$($(expr match "$(cat /tmp/VERSION)" ".*KERNEL=\(.*\)"))"

		log "${TYPE}" "Installing Kernel-Image. This process may take 3 minutes."
		log "${TYPE}" "Do NOT turn off your beroNet VoIP Gateway!"
		log "${TYPE}" "Updating Kernel-Image from Version ${VERSION_INSTALLED} to ${VERSION_TOINSTALL}."

		# log update-operations
		echo "[${TYPE}] $(date): ${VERSION_INSTALLED} => ${VERSION_TOINSTALL}" >> /usr/conf/update.log
		mv /tmp/VERSION /usr/conf/VERSION.kernel

		# prepare the next reboot
		${BEROCONF} set root boot_fwupdate 0
		echo "1" > /tmp/reboot_after_update

		sleep 2
		dd if=/tmp/zImage of=/dev/mtdblock4
	;;
	rootfs)
		VERSION_INSTALLED="unknown"
		if [ -f /VERSION ]; then
			VERSION_INSTALLED=$(expr match "$(cat /VERSION)" ".*ROOTFS=\(.*\)")
		fi
		VERSION_TOINSTALL=$(expr match "$(cat /tmp/VERSION)" ".*ROOTFS=\(.*\)")

		echo "[${TYPE}] $(date): ${VERSION_INSTALLED} => ${VERSION_TOINSTALL}" >> /usr/conf/update.log

		if [ -f /etc/ssh_host_key ]; then
			log "${TYPE}" "Saving SSH-Leys to /usr/conf."
			[ ! -d /usr/conf/ssh ] && mkdir /usr/conf/ssh
			cp -a /etc/ssh_host* /usr/conf/ssh/
		fi

		log "${TYPE}" "Stopping services."
		/etc/init.d/S90app stop
		/etc/init.d/S60httpd stop

		# prepare the next reboot
		${BEROCONF} set root boot_fwupdate 0

		log "${TYPE}" "Loading berofix module."
		insmod /usr/local/modules/berofix.ko

		log "${TYPE}" "Extracting RootFS-Image."
		rm -rf /tmp/newroot
		mkdir /tmp/newroot || exit 1
		cd /tmp/newroot
		tar zxf ${image_dir}/${file} root.jffs2 || exit 1

		log "${TYPE}" "Saving RootFS to flash memory."
		dd if=root.jffs2 of=/dev/mtdblock5

		log "${TYPE}" "rebooting"
		dd if=/proc/RESET of=/dev/null
		;;
	userappfs)
		/usr/sbin/userapp_pkg_install.sh ${file}
		;;
	appfs)
		# Put on a LED-Show for the custumers!
		if [ -f /sys/class/beroNet_basicfunctions/led_mode ]; then
			echo 3 > /sys/class/beroNet_basicfunctions/led_mode
		fi

		#check for existence of new berofix-driver
		lsmod|grep berofix-driver &> /dev/null
		RET1=$?
		unload_flag=0
		#if it isn't loaded, load it
		if [ $RET1 != 0 ];then
			/sbin/insmod /lib/modules/berofix-driver.ko
			unload_flag=1
		fi
		
		DOWNGRADE_FOR_ANY_FW_ALLOWED=1
		if [ -f /sys/class/beronet/gateway/type ]; then
			GWTYPE=`cat /sys/class/beronet/gateway/type`
			#this
			if [ "$GWTYPE" -ge 30 ]; then
				DOWNGRADE_FOR_ANY_FW_ALLOWED=0
			fi
		fi
		
		if [ $unload_flag = 1 ]; then
			/sbin/rmmod berofix-driver.ko
		fi
		
		#we are on new hardware, we are not to allowed to dowgrade to FWs less than 16.11
		if [ $DOWNGRADE_FOR_ANY_FW_ALLOWED = 0 ]; then
			if [ -z "${APPFS}" ]; then
				APPFS=1
			fi
			if [ ${APPFS} -lt 17 ]; then
				#we allow installing FW 16.11 to 16.19
				echo ${file} | grep 'appfs-16.1[1-9]' &> /dev/null
				if [ $? != 0 ];then
					log "${TYPE}" "Downgrade to ${file} not allowed on this HW!"
					rm -f ${PIDFILE}
					exit 1
				fi
			fi
		fi
		

		DOWNGRADE_IND=$(echo ${file} | grep "downgrade")
		if [ -z "${DOWNGRADE_IND}" ]; then
			log "${TYPE}" "Checking the Integrity of Archive."
			cd /tmp && tar zxf ${image_dir}/${file} content.txt 2>/dev/null
			if [ ! -f /tmp/content.txt ]; then
				log "${TYPE}" "FAIL: Archive not compatible!"
				rm -f ${PIDFILE}
				exit 1
			fi
			tar tzf ${image_dir}/${file} > /tmp/compare-content.txt
			for i in $(cat /tmp/content.txt | sed "s/\/$//" ); do
				if ! grep "${i}" compare-content.txt > /dev/null; then
					log "${TYPE}" "FAIL: File ${i} missing in archive, leaving!"
					rm -f ${PIDFILE}
					exit 1
				fi
			done
		fi

		rm -rf /tmp/content.txt
                rm -rf /tmp/compare-content.txt

		log "${TYPE}" "Checking Versions of installed Kernel and Firmware."
		if [ -f /usr/conf/VERSION.kernel ]; then
			KERNEL=$(expr match "$(cat /usr/conf/VERSION.kernel)" ".*VERSION=\([0-9]*\)")
		else
			KERNEL=1
		fi
		if [ -f /usr/conf/VERSION.fpga ]; then
			FPGA=$(expr match "$(cat /usr/conf/VERSION.kernel)" ".*VERSION=\([0-9]*\)")
		else
			FPGA=8
		fi
		if [ -f /usr/conf/VERSION.firmware ]; then
			FIRMWARE=$(expr match "$(cat /usr/conf/VERSION.firmware)" ".*VERSION=\([0-9]*\)")
		else
			FIRMWARE=1
		fi

		if [ -f /VERSION ]; then
			ROOTFS=$(expr match "$(cat /VERSION)" ".*ROOTFS=\([0-9]*\)")
		else
			ROOTFS=1
		fi

		version_check_fail=0
		if [ "${NEED_KERNEL}" -gt "${KERNEL}" ]; then
			log "${TYPE}" "FAIL: Please update Kernel to version ${NEED_KERNEL} and try again!"
			version_check_fail=1
		fi
		if [ "${NEED_FPGA}" -gt "${FPGA}" ]; then
			log "${TYPE}" "FAIL: Please update FPGA to version ${NEED_FPGA} and try again!"
			version_check_fail=1
		fi
		if [ "${NEED_FIRMWARE}" -gt "${FIRMWARE}" ]; then
			log "${TYPE}" "FAIL: Please update Firmware to version ${NEED_FIRMWARE} and try again!"
			version_check_fail=1
		fi
		if [ "${NEED_ROOTFS}" -gt "${ROOTFS}" ]; then
			log "${TYPE}" "FAIL: Please update RootFS to version ${NEED_ROOTFS} and try again!"
			version_check_fail=1
		fi

		if [ "${version_check_fail}" = "1" ]; then
			tmp_cleanup
			rm -f ${PIDFILE}
			exit 1
		fi

		VERSION_INSTALLED="$(cat /usr/local/FILENAME)"

		if [ "$(${BEROCONF} get root boot_fwupdate)" != "1" ]; then
			/etc/init.d/S45cron stop > /dev/null

        	        log "${TYPE}" "Stopping UserAppFS-Services."
                	/etc/init.d/S91userapp stop > /dev/null

			log "${TYPE}" "Stopping AppFS-Services."
			/etc/init.d/S90app stop > /dev/null
		fi

		log "${TYPE}" "Remounting AppFS partition (rw)."
		mount -o remount,rw /usr/local || exit 1

		log "${TYPE}" "Executing pre_remove-Script."
		[ -x /usr/local/setup/pre_remove.sh ] && /usr/local/setup/pre_remove.sh

		log "${TYPE}" "Removing old Version-File."
		[ -f /usr/local/FILENAME ] && rm -f /usr/local/FILENAME

		log "${TYPE}" "Cleaning up AppFS-Partition."
		cd /usr/local && ls | grep -v lost | xargs rm -rf

		log "${TYPE}" "Extracting new AppFS-Image."
		cd /usr/local && tar zxf ${image_dir}/${file} || exit 1

		log "${TYPE}" "Setting Permissions."
		chown -R root:root /usr/local/*

		log "${TYPE}" "Writing new Version-File."
		echo "${file}" > /usr/local/FILENAME

		if [ ! -z "${DOWNGRADE_IND}" ]; then
			log "${TYPE}" "Cleaning up downgrade-incompatible configuration files."
			rm -f /usr/conf/isgw.gsm
		fi

		sync; sleep 1; sync

                log "${TYPE}" "Removing uploaded AppFS-Image."
                rm -f ${image_dir}/${file}
		if [ -f /home/admin/${file} ]; then
			mount -oremount,rw /home/admin
			rm -f /home/admin/${file}
			mount -oremount,ro /home/admin
		fi

		log "${TYPE}" "Executing post_install-Script."
		[ -x /usr/local/setup/post_install.sh ] && /usr/local/setup/post_install.sh

		log "${TYPE}" "Remounting AppFS-Partition (ro)."
		cd / && mount -o remount,ro /usr/local

                sync

		${BEROCONF} set root boot_fwupdate 0
		${BEROCONF} set root app-image $(cat /usr/local/FILENAME)

		VERSION_TOINSTALL="$(cat /usr/local/FILENAME)"
		echo "[${TYPE}] $(date): ${VERSION_INSTALLED} => ${VERSION_TOINSTALL}" >> /usr/conf/update.log

		if [ ! -f /tmp/reboot_after_update ]; then
			# Remove integrity-marker.
			if [ -f /tmp/enable_recovery ]; then
				rm -f /tmp/enable_recovery
			fi

			# End LED-Show.
			if [ -f /sys/class/beroNet_basicfunctions/led_mode ]; then
				echo 0 > /sys/class/beroNet_basicfunctions/led_mode
			fi

			log "${TYPE}" "Restarting RootFS-Services."
			/etc/init.d/S40network restart > /dev/null
			/etc/init.d/S41ncs restart > /dev/null
			/etc/init.d/S45cron start > /dev/null

			log "${TYPE}" "Restarting AppFS-Services."
			for i in /usr/local/init/S??* ; do
				# Ignore dangling symlinks (if any).
				[ ! -f "${i}" ] && continue
				if [ "$i" = "/usr/local/init/S99upgrade-firmware" ] || [ "$i" = "/usr/local/init/S02update-conf" ]; then
					continue
				fi
				${i} start
			done
			${BEROCONF} set root app-status running

			log "${TYPE}" "Restarting UserAppFS-Services."
                	/etc/init.d/S91userapp start > /dev/null
		fi
		;;
esac

if [ -f /tmp/reboot_after_update ]; then
	log ${TYPE} "Rebooting beroNet VoIP Gateway to complete the Update."
fi

sync
sleep 1

rm -f ${PIDFILE}

exit 0
