#!/bin/bash
#
# Usage: install_daemon.sh <imagefile>
#
# VERSION=30
# CHANGES="update the script to install appfs firmware independently where mtdblock6 is mounted"

# constants #
BEROCONF=/usr/fallback/beroconf
PIDFILE=/var/run/install_daemon.pid

file=${1}
image_dir=/tmp/images

# functions #
function log {

	prefix=${1:common}
	logfile=/usr/conf/permlog/install-${prefix}.log
	pipe=/tmp/install_pipe

	if [ -p ${pipe} ]; then
		echo "[install_${prefix}] $(date): ${2}" &>${pipe}
	fi

	echo "[install_${prefix}] $(date): ${2}" >> ${logfile}
}

function fix_old_image_dir {
	mount -o remount,rw /
	if [ -d /mnt/images ]; then
		rm -rf /mnt/images
	fi
	ln -s /tmp /mnt/images
	mount -o remount,ro /
}

function tmp_cleanup {
	for i in "VERSION content.txt compare-content.txt"; do
		rm -f ${i}
	done
}

function clean_openssl_library {
	library_type=$1
	library_name=$(cd /home/admin/lib && ls $library_type.so.*)
	if [ ! -z $library_name ];
	then
		mount -o remount,rw /home/admin
		rm -f /home/admin/lib/$library_type.*
		mount -o remount,ro /home/admin
		sync; sleep 1; sync;

		mount -o remount,rw /
		rm -f /usr/lib/$library_name
		mount -o remount,ro /
		sync; sleep 1; sync;
	fi
}

function exit_daemon {
	rm -f /tmp/VERSION
	rm -f ${PIDFILE}
	sleep 5
	echo -n 2 > /sys/class/beronet/gateway/led_program
	exit 1
}

# main #
if [ ! -d ${image_dir} ]; then
	mkdir -p ${image_dir}
fi

if [ -z "${file}" ] ; then
	log "init" "ERROR: No filename given, leaving."
	exit_daemon
fi

if [ ! -f "${image_dir}/${file}" ] ; then
	log "init" "ERROR: File ${image_dir}/${file} not found, leaving."
	exit_daemon
fi

if [ ! -L /mnt/images ]; then
	fix_old_image_dir
fi

source /usr/local/VERSION
APPFS_INSTALLED=${APPFS}
[ -z ${APPFS_INSTALLED} ] && APPFS_INSTALLED=1

if [ -z "${OPENSSL}" ]; then
	OPENSSL_VERSION_INSTALLED=0
else
	OPENSSL_VERSION_INSTALLED=${OPENSSL}
	unset OPENSSL 
fi

if [ ! -f /tmp/VERSION ];
then
	## should not happen since VERSION is untared from the fallback/install.sh script
	tar zxf ${image_dir}/${file} VERSION -C /tmp
	if [ ! -f /tmp/VERSION ];
	then
		tar xzf ${image_dir}/${file} usr/local/VERSION -C /tmp
		mv /tmp/usr/local/VERSION /tmp/
	fi
fi
source /tmp/VERSION
[ -z ${APPFS} ] && APPFS=0

# check if TYPE-field is safe.
case "${TYPE}" in
	fpga|firmware|kernel|rootfs|appfs|appfsV2|userappfs)
		;;
	*)
		log "unknown" "ERROR: This package does not contain a valid type, leaving."
		exit_daemon
		;;
esac

FS_NAME="AppFS"

# remove previous log-file
rm -f /usr/conf/permlog/install-${TYPE}.log

log "${TYPE}" "Installing '${file}'."
case "${TYPE}" in
	fpga)
		# Put on a LED-Show for the customers!
		if [ -f /sys/class/beronet/gateway/led_program ]; then
			echo -n 3 > /sys/class/beronet/gateway/led_program
		fi

		log "${TYPE}" "Unpacking FPGA-Image."
		cd /tmp && tar zxf ${image_dir}/${file} fpga.img
		if [ ! -f /tmp/fpga.img ] ; then
			log "${TYPE}" "ERROR: Could not unpack the FPGA-Image, leaving."
			exit_daemon
		fi

		# retrieve Version-Information
		VERSION_INSTALLED=$(expr match "$(cat /sys/class/beronet/fpga/version)" "\([0-9]*\)")
		VERSION_TOINSTALL="$(expr match "$(cat /tmp/VERSION)" ".*FPGA=\(.*\)")"

		# tell the user about it
		log "${TYPE}" "Installing FPGA-Image. This process may take 3 minutes."
		log "${TYPE}" "Do NOT turn off your beroNet VoIP Gateway!"
		log "${TYPE}" "Updating FPGA-Image from Version ${VERSION_INSTALLED} to ${VERSION_TOINSTALL}."

		# log update-operations
		echo "[${TYPE}] $(date): ${VERSION_INSTALLED} => ${VERSION_TOINSTALL}" >> /usr/conf/update.log

		# activate write-mode on fpga-eeprom
		echo 1 > /sys/class/beronet/fpga/eeprom
		if [ "$(cat /sys/class/beronet/fpga/eeprom)" -ne "1" ]; then
			log "${TYPE}" "ERROR: Could not enable write-access to FPGA-eeprom, leaving."
			exit_daemon
		fi

		# prepare the next reboot
		${BEROCONF} set root boot_fwupdate 0
		echo "1" > /tmp/reboot_after_update

		# Perform the Update
		sleep 2
		dd if=/tmp/fpga.img of=/dev/bfpga bs=512k count=1
		;;
	firmware)
		# Put on a LED-Show for the customers!
		if [ -f /sys/class/beronet/gateway/led_program ]; then
			echo -n 3 > /sys/class/beronet/gateway/led_program
		fi

		log "${TYPE}" "Unpacking Firmware."
		cd /tmp && tar zxf ${image_dir}/${file} firmware.axf
		if [ ! -f /tmp/firmware.axf ]; then
			log "${TYPE}" "ERROR: Could not unpack the Firmware-Image, leaving."
			exit_daemon
		fi

		# retrieve Version-Information
		VERSION_INSTALLED="unknown"
		if [ -f /usr/conf/VERSION.firmware ]; then
			VERSION_INSTALLED="$(expr match "$(cat /usr/conf/VERSION.firmware)" ".*FIRMWARE=\(.*\)")"
		fi
		VERSION_TOINSTALL="$(expr match "$(cat /tmp/VERSION)" ".*FIRMWARE=\(.*\)")"

		log "${TYPE}" "Installing Firmware-Image. This process may take 3 minutes."
		log "${TYPE}" "Do NOT turn off your beroNet VoIP Gateway!"
		log "${TYPE}" "Updating Firmware-Image from Version ${VERSION_INSTALLED} to ${VERSION_TOINSTALL}."

		# prepare the next reboot
		${BEROCONF} set root boot_fwupdate 0
		echo "1" > /tmp/reboot_after_update

		# log update-operations
		echo "[${TYPE}] $(date): ${VERSION_INSTALLED} => ${VERSION_TOINSTALL}" >> /usr/conf/update.log
		mv /tmp/VERSION /usr/conf/VERSION.firmware

		# Perform the Update
		sleep 2
		dd if=/tmp/firmware.axf of=/dev/mtdblock3
	;;
	kernel)
		# Put on a LED-Show for the customers!
		if [ -f /sys/class/beronet/gateway/led_program ]; then
			echo -n 3 > /sys/class/beronet/gateway/led_program
		fi

		log "${TYPE}" "Unpacking Kernel-Image."
		cd /tmp && tar zxf ${image_dir}/${file} zImage
		if [ ! -f /tmp/zImage ]; then
			log "${TYPE}" "ERROR: Could not unpack the Kernel-Image, leaving."
			exit_daemon
		fi

		# Retrieve Version-Information
		VERSION_INSTALLED="unknown"
		if [ -f /usr/conf/VERSION.${TYPE} ]; then
			VERSION_INSTALLED="$($(expr match "$(cat /usr/conf/VERSION.kernel)" ".*KERNEL=\(.*\)"))"
		fi
		VERSION_TOINSTALL="$($(expr match "$(cat /tmp/VERSION)" ".*KERNEL=\(.*\)"))"

		log "${TYPE}" "Installing Kernel-Image. This process may take 3 minutes."
		log "${TYPE}" "Do NOT turn off your beroNet VoIP Gateway!"
		log "${TYPE}" "Updating Kernel-Image from Version ${VERSION_INSTALLED} to ${VERSION_TOINSTALL}."

		# log update-operations
		echo "[${TYPE}] $(date): ${VERSION_INSTALLED} => ${VERSION_TOINSTALL}" >> /usr/conf/update.log
		mv /tmp/VERSION /usr/conf/VERSION.kernel

		# prepare the next reboot
		${BEROCONF} set root boot_fwupdate 0
		echo "1" > /tmp/reboot_after_update

		sleep 2
		dd if=/tmp/zImage of=/dev/mtdblock4
	;;
	rootfs)
		# Put on a LED-Show for the customers!
		if [ -f /sys/class/beronet/gateway/led_program ]; then
			echo -n 3 > /sys/class/beronet/gateway/led_program
		fi

		VERSION_INSTALLED="unknown"
		if [ -f /VERSION ]; then
			VERSION_INSTALLED=$(expr match "$(cat /VERSION)" ".*ROOTFS=\(.*\)")
		fi
		VERSION_TOINSTALL=$(expr match "$(cat /tmp/VERSION)" ".*ROOTFS=\(.*\)")

		echo "[${TYPE}] $(date): ${VERSION_INSTALLED} => ${VERSION_TOINSTALL}" >> /usr/conf/update.log

		if [ -f /etc/ssh_host_key ]; then
			log "${TYPE}" "Saving SSH-Leys to /usr/conf."
			[ ! -d /usr/conf/ssh ] && mkdir /usr/conf/ssh
			cp -a /etc/ssh_host* /usr/conf/ssh/
		fi

		log "${TYPE}" "Stopping services."
		/etc/init.d/S90app stop
		/etc/init.d/S60httpd stop

		# prepare the next reboot
		${BEROCONF} set root boot_fwupdate 0

		log "${TYPE}" "Loading berofix module."
		insmod /usr/local/modules/berofix.ko

		log "${TYPE}" "Extracting RootFS-Image."
		rm -rf /tmp/newroot
		mkdir /tmp/newroot || exit 1
		cd /tmp/newroot
		tar zxf ${image_dir}/${file} root.jffs2 || exit 1

		log "${TYPE}" "Saving RootFS to flash memory."
		dd if=root.jffs2 of=/dev/mtdblock5

		log "${TYPE}" "rebooting"
		echo 1 > /sys/class/beronet/fpga/reset
		;;
	userappfs)
		/usr/sbin/userapp_pkg_install.sh ${file}
		rm -f ${PIDFILE}
		;;
	appfs|appfsV2)
		###### DOWNGRADE SCENARII #######

		# downgrades to appfs older than 17 is no longer supported
		if [ ${APPFS} -lt 17 ]; then
			if [ ! -f /tmp/DOWNGRADED ]; then
				log "${TYPE}" "Downgrading to a Firmware older than 16.11 is not supported."
				rm -f /tmp/DOWNGRADED
				sleep 10
				tmp_cleanup
				exit_daemon
			fi
			log "${TYPE}" "Installing older Firmware. Basic downgrade was made, continuing with Firmware install."
		fi

		# downgrade from firmware >= 21 to firmware < 21
		# check if lan-dhcp is enabled on lan-wan mode
		IS_DOWNGRADE=0
		if [ ${APPFS} -lt 21 ]; then
			IS_DOWNGRADE=1
		elif [[ ${APPFS} -eq 21 ]] && [[ "${MINOR}" == "01" ]]; then
			if [[ ! -z ${RC} ]] && [[ ${RC} -lt 43 ]]; then
				IS_DOWNGRADE=1
			fi	
		fi
		if [ ${IS_DOWNGRADE} -eq 1 ]; then
			if [ -f /home/admin/bin/php-cgi ]; then
				echo '1' > /tmp/restoring_phpcgi_rootfs
				${BEROCONF} set root downgrading-required 1
				log "${TYPE}" "Inappropriate RootFs to a firmware downgrade"
				sleep 5
				exit_daemon
			fi
			LAN_MODE=$(${BEROCONF} get root lan-mode | grep -v failed)
			LAN_DHCP=$(${BEROCONF} get root lan-dhcp | grep -v failed)
			if [[ "${LAN_MODE}" == "lan-wan" ]] && [[ "${LAN_DHCP}" -eq "1" ]]; then
				echo '1' > /tmp/switch_network_configuration
				log "${TYPE}" "Inappropriate Network configuration to a firmware downgrade"
				sleep 5
				exit_daemon
			fi
		## upgrade from 21.X to higher.
		## since libcrypto and libssl belongs to /home/admin, the mtdblock7 space size is not correclty updated (remount impossible) (since lighttpd is running)
		## so a size tampon is required to have a proper installation
		else
			available=$(df | grep mtdblock7 | awk '{ print $4; }')
			if [ "${available}" -lt "1300" ]; then
				echo '1' > /tmp/homeadmin_notenough_space
				log "${TYPE}" "UserApps-Storage: not enough space"
				sleep 5
				exit_daemon
			fi
		fi

		###############	

		## check enough space on the /home/admin space
		if [[ ! -z ${OPENSSL} ]] && [[ "${OPENSSL_VERSION_INSTALLED}" -eq "0" ]];
		then
			available=$(df | grep mtdblock7 | awk '{ print $4; }')
			if [ "${available}" -lt "3500" ];
			then
				echo '1' > /tmp/clean_home_admin
				log "${TYPE}" "Not enough space. Please remove some apps."
				sleep 10
				exit_daemon
			fi
		fi

		# Put on a LED-Show for the customers!
		if [ -f /sys/class/beronet/gateway/led_program ]; then
			echo -n 3 > /sys/class/beronet/gateway/led_program
		fi

		# Checking firmware integrity
		log "${TYPE}" "Checking firmware integrity"
		CONTENT_PATH=/tmp/usr/local
		if ! tar xzf ${image_dir}/${file} usr/local/content.txt -C /tmp 2>/dev/null ; then
			## old firmware
			CONTENT_PATH=/tmp
			if ! tar xzf ${image_dir}/${file} content.txt -C /tmp 2>/dev/null; then
				log "${TYPE}" "FAIL: File content.txt missing in archive. Leaving !"
				exit_daemon
			fi
		fi
		tar tzf ${image_dir}/${file} > /tmp/compare-content.txt
		for i in $(cat ${CONTENT_PATH}/content.txt | sed "s/\/$//"); do
			if ! grep "${i}" /tmp/compare-content.txt >/dev/null; then
				log "${TYPE}" "FAIL: File ${i} missing in archive. Leaving !"
				sleep 5
				exit_daemon
			fi
		done
		rm -rf ${CONTENT_PATH}/content.txt
		rm -rf /tmp/compare-content.txt
		unset CONTENT_PATH

		# other checks
		log "${TYPE}" "Checking Versions of installed Kernel and Firmware."
		if [ -f /usr/conf/VERSION.kernel ]; then
			KERNEL=$(expr match "$(cat /usr/conf/VERSION.kernel)" ".*VERSION=\([0-9]*\)")
		else
			KERNEL=1
		fi
		if [ -f /sys/class/beronet/fpga/version ]; then
			FPGA=$(cat /sys/class/beronet/fpga/version)
		else
			FPGA=8
		fi
		if [ -f /usr/conf/VERSION.firmware ]; then
			FIRMWARE=$(expr match "$(cat /usr/conf/VERSION.firmware)" ".*VERSION=\([0-9]*\)")
		else
			FIRMWARE=1
		fi

		if [ -f /pkginfo/VERSION.rootfs ]; then
			ROOTFS=$(expr match "$(cat /pkginfo/VERSION.rootfs)" ".*PKG_VERSION=\([0-9]*\)")
		else
			ROOTFS=1
		fi

		version_check_fail=0
		if [ "${NEED_KERNEL}" -gt "${KERNEL}" ]; then
			log "${TYPE}" "FAIL: Please update Kernel to version ${NEED_KERNEL} and try again!"
			version_check_fail=1
		fi
		if [ "${NEED_FPGA}" -gt "${FPGA}" ]; then
			log "${TYPE}" "FAIL: Please update FPGA to version ${NEED_FPGA} and try again!"
			version_check_fail=1
		fi
		if [ "${NEED_FIRMWARE}" -gt "${FIRMWARE}" ]; then
			log "${TYPE}" "FAIL: Please update Firmware to version ${NEED_FIRMWARE} and try again!"
			version_check_fail=1
		fi
		if [ "${NEED_ROOTFS}" -gt "${ROOTFS}" ]; then
			log "${TYPE}" "FAIL: Please update RootFS to version ${NEED_ROOTFS} and try again!"
			version_check_fail=1
		fi

		if [ "${version_check_fail}" = "1" ]; then
			log "${TYPE}" "ERROR: checking the VERSION failed"
			sleep 10
			tmp_cleanup
			exit_daemon
		fi

		if [ "$(${BEROCONF} get root boot_fwupdate)" != "1" ]; then
			/etc/init.d/S45cron stop > /dev/null

			log "${TYPE}" "Stopping UserAppFS-Services."
			/etc/init.d/S91userapp stop > /dev/null

			log "${TYPE}" "Stopping ${FS_NAME}-Services."
			/etc/init.d/S90app stop > /dev/null
		fi

		log "${TYPE}" "Remounting ${FS_NAME}-partition (rw)."
		if ! mount -o remount,rw /usr/local ;
		then
			log "${TYPE}" "ERROR: Remounting ${FS_NAME}-partition (rw) failed"
			exit_daemon
		fi

		VERSION_INSTALLED="$(cat /usr/local/FILENAME)"

		## remove variables from recovery-mode / clean up beroconf database
		${BEROCONF} delete root boot_recoverymode
		${BEROCONF} delete root appfs-reset
		${BEROCONF} delete root rootfs-reset
		${BEROCONF} delete root userappfs-reset

		## clean up appfs-partition
		log "${TYPE}" "Cleaning up ${FS_NAME}-Partition."
		cd /usr/local && ls | grep -v lost | xargs rm -rf

		## remount the /usr/local space to update the space size
		mount -o remount,ro /usr/local
		sync; sleep 1; sync;
		mount -o remount,rw /usr/local

		## extracting the appfs depending openssl version into the appfs firmware
		log "${TYPE}" "Extracting new ${FS_NAME}-Image."
		if [ -z "${OPENSSL}" ];
		then
			## PLEASE NOTE: 
			##	o Normal case: untaring old firmware to the /usr/local space should never failed
			##	o Developement case: depending the local firmware, untaring it can fail. Since the old VERSION file has been removed
			##	  from the /usr/local space, update it manually to follow the expected scenario (old classic installation or downgrade)

			if [ "${OPENSSL_VERSION_INSTALLED}" -eq "0" ];
			then
				## old classic installation ##
				## we untar the full apffs into the /usr/local space
				if ! tar xzf ${image_dir}/${file} -C /usr/local >/dev/null ;
				then
					## this fail should never happen
					mount -o remount,ro /usr/local
					log "${TYPE}" "Extracting ${FS_NAME}-Image failed: /usr/local space full"
					exit_daemon
				fi
			else
				## downgrade scenario ##
				## we go back to the old configuration using the old openssl libraries (0.9.7 and 0.9.8)
				if ! tar xzf ${image_dir}/${file} -C /usr/local >/dev/null ;
				then
					## this fail should never happen
					mount -o remount,ro /usr/local
					log "${TYPE}" "Extracting ${FS_NAME}-Image failed: /usr/local space full"
					exit_daemon
				fi

				## we fake the VERSION.lighttpd and VERSION.curl decreasing their
				## value to install the lighttd binary and curl libraries during boot/update
				## (an overwrite of the previous PKG_VERSION downgrade => avoid unexpected behavior)
				mount -o remount,rw /
				sed -i 's/PKG_VERSION=.*/PKG_VERSION=1/g' /pkginfo/VERSION.curl
				sed -i 's/PKG_VERSION=.*/PKG_VERSION=1/g' /pkginfo/VERSION.lighttpd
				mount -o remount,ro /
				sync; sleep 1; sync

				## remove libcrypto and libssl libraries (binary and symlink)
				clean_openssl_library "libcrypto"
				clean_openssl_library "libssl"

				## remove conf/rootfs from /home/admin space
				if [ -d /home/admin/conf ];
				then
					mount -o remount,rw /home/admin
					rm -rf /home/admin/conf
					mount -o remount,ro /home/admin
					sync; sleep 1; sync;
				fi
			fi
		else
			## remove libcrypto and libssl libraries (binary and symlink)
			clean_openssl_library "libcrypto"
			clean_openssl_library "libssl"

			## clean conf from /home/admin
			mount -o remount,rw /home/admin
			rm -rf /home/admin/conf
			mount -o remount,ro /home/admin	
			sync; sleep 1; sync;

			## update /usr/local space
			mount -o remount,rw /usr/local
			tar xzf ${image_dir}/${file} usr/local/* -C / >/dev/null
			mount -o remount,ro /usr/local
			sync; sleep 1; sync;

			## update /home/admin space
			mount -o remount,rw /home/admin
			tar xzf ${image_dir}/${file} home/admin/* -C / >/dev/null
			mount -o remount,ro /home/admin
			sync; sleep 1; sync

			## create symlink to libcrypto and libssl
			mount -o remount,rw /
			ln -sf /home/admin/lib/libcrypto.so.* /usr/lib
			ln -sf /home/admin/lib/libssl.so.* /usr/lib
			mount -o remount,ro /
			sync; sleep 1; sync

			## untar /rootfs into /tmp directory
			tar xzf ${image_dir}/${file} conf/rootfs/* -C /tmp >/dev/null

			## move VERSION files to /usr/local/conf/rootfs directory 
			mount -o remount,rw /usr/local
			mkdir -p /usr/local/conf/rootfs
			mv /tmp/conf/rootfs/VERSION.* /usr/local/conf/rootfs/

			mount -o remount,rw /home/admin
			mv /home/admin/conf/rootfs/VERSION.* /usr/local/conf/rootfs/
			mount -o remount,ro /home/admin
		fi

		log "${TYPE}" "Setting Permissions."
		chown -R root:root /usr/local/*
		sync; sleep 1; sync

		log "${TYPE}" "Removing uploaded ${FS_NAME}-Image."
 		rm -f ${image_dir}/${file}

		## general downgrade scenario: the new major version is lower than the one installed
		## we decrease the PKG_VERSION value for the /pkginfo/VERSION.* files
		if [ "${APPFS}" -lt "${APPFS_INSTALLED}" ]; then
			mount -o remount,rw /
			for file in $(ls /pkginfo/VERSION.*); do
				/bin/sed -i 's/PKG_VERSION=.*/PKG_VERSION=1/g' $file
			done
			mount -o remount,ro /
			sync; sleep 1; sync;
		fi

################ WHY THIS SECTION -> can be removed ??? ##########
		if [ -f /home/admin/${file} ]; then
			mount -oremount,rw /home/admin
			rm -f /home/admin/${file}
			mount -oremount,ro /home/admin
		fi
#################################################################

		log "${TYPE}" "Executing post_install-Script."
		[ -x /usr/local/setup/post_install.sh ] && /usr/local/setup/post_install.sh

		log "${TYPE}" "Remounting ${FS_NAME}-Partition (ro)."
		cd / && mount -o remount,ro /usr/local 

		sync

		VERSION_TOINSTALL="$(cat /usr/local/FILENAME)"

		${BEROCONF} set root boot_fwupdate 0
		${BEROCONF} set root app-image ${VERSION_TOINSTALL}

		echo "[${TYPE}] $(date): ${VERSION_INSTALLED} => ${VERSION_TOINSTALL}" >> /usr/conf/update.log

		#we always reboot after an appfs update now
		echo 1 > /tmp/reboot_after_update
		;;
esac

if [ -f /tmp/reboot_after_update ]; then
	log ${TYPE} "Rebooting beroNet VoIP Gateway to complete the Update."
fi
sync
sleep 1
rm -f ${PIDFILE}
exit 0
