#!/bin/bash
#
# VERSION=4
# CHANGES="change back SBCFS to AppFS"
#
# Description: This script installs UserApps to the beroNet VoIP Gateway.
# Author: Florian Kraatz <fk@beronet.com>
#

# functions #
function log {
	echo "[userapp_install] $(date): ${1}" | tee -a /usr/conf/permlog/install-userapp.log
}

function tmp_cleanup {
	for i in "VERSION content.txt compare-content.txt"; do
		rm -f ${i}
	done
}

# constants #
export PATH="/bin:/sbin:/usr/bin:/usr/sbin:/usr/local/bin:/usr/local/sbin:/usr/fallback"

IMAGE_DIR=/tmp/images
USERAPP_ROOT=/apps
USERAPP_HOME=/home/admin

# parameters #
pkg_file=${1}
no_start=${2}

# main #

if [ ! -d /usr/conf/permlog ]; then
	mkdir -p /usr/conf/permlog
fi

if [ -f /usr/conf/permlog/install-userapp.log ]; then
	rm -f /usr/conf/permlog/install-userapp.log
fi

if [ -z "${pkg_file}" ]; then
	log "ERROR: No file given, leaving."
	exit 1
fi

if [ ! -f "${IMAGE_DIR}/${pkg_file}" ] ; then
	log "ERROR: Package ${IMAGE_DIR}/${pkg_file} not found, leaving."
	exit 2
fi

cd /tmp && tar zxf ${IMAGE_DIR}/${pkg_file} VERSION
. /tmp/VERSION

if [ -z "${TYPE}" ] || [ "${TYPE}" != "userappfs" ]; then
	log "ERROR: Package has no or an invalid type defined, leaving."
	exit 3
fi

if [ -z ${NAME} ]; then
	log "ERROR: Name of this UserApp is empty, leaving."
	exit 4
fi

log "Cleaning up Temp-Directory."
tmp_cleanup

log "Name:        ${NAME}"
log "Version:     ${VERSION}"
log "Description: ${DESCRIPTION}"

log "Checking Integrity of Archive."
cd /tmp
tar zxf ${IMAGE_DIR}/${pkg_file} content.txt 2> /dev/null
if [ ! -f /tmp/content.txt ]; then
	log "ERROR: Archive is not compatible!"
	exit 5
fi

tar tzf ${IMAGE_DIR}/${pkg_file}  > /tmp/compare-content.txt
for i in $(cat /tmp/content.txt | sed "s/\/$//" )  ; do
	if ! grep "${i}" compare-content.txt > /dev/null ; then
		log "ERROR: File '${i}' missing in archive, leaving."
		exit 6
	fi
done

#log "Checking Versions:"
## check kernel
#if [ -f /usr/conf/VERSION.kernel ] ; then
#	KERNEL=$(expr match "$(cat /usr/conf/VERSION.kernel)" ".*VERSION=\([0-9]*\)")
#else
#	KERNEL=1
#fi
#if [ ${NEED_KERNEL} -gt ${KERNEL} ] ; then
#	log "ERROR: Please update Kernel to Version ${NEED_KERNEL} and try again!"
#	exit 7
#fi
#
## check rootfs
#if [ -f /VERSION ] ; then
#	ROOTFS=$(expr match "$(cat /VERSION)" ".*ROOTFS=\([0-9]*\)")
#else
#	ROOTFS=1
#fi
#if [ ${NEED_ROOTFS} -gt ${ROOTFS} ] ; then
#	log "ERROR: Please update RootFS to Version ${NEED_ROOTFS} and try again!"
#	exit 8
#fi
#
## check firmware
#if [ -f /usr/conf/VERSION.firmware ]; then
#	FIRMWARE=$(expr match "$(cat /usr/conf/VERSION.firmware)" ".*VERSION=\([0-9]*\)")
#else
#	FIRMWARE=1
#fi
#if [ ${NEED_FIRMWARE} -gt ${FIRMWARE} ]; then
#	log "ERROR: Please update Firmware to Version ${NEED_FIRMWARE} and try again!"
#	exit 9
#fi

# check AppFS-Version
if [ -f /usr/local/VERSION ]; then
	APPFS_FIL="$(cat /usr/local/VERSION)"
	APPFS=$(expr match "${APPFS_FIL}" ".*AppFS=\([0-9]*\)")
	unset APPFS_FIL
else
	APPFS=1
fi
if [ ${NEED_APPFS} -gt ${APPFS} ]; then
	log "ERROR: Please update AppFS to Version ${NEED_APPFS} and try again!"
	exit 10
fi

# if this is not an upgrade, check if there is enough space left
if [ ! -d /apps/${NAME} ]; then

	if [ -z "${SIZE}" ]; then
		let SIZE="${SLOTS}*1024"
	else
		let SIZE="${SIZE}/1024"
	fi

	SIZE_FREE=$(df | grep ${USERAPP_HOME} | awk '{ print $4; }')

	if [ ${SIZE} -ge ${SIZE_FREE} ]; then
		log "ERROR: Installing this app will exceed free disk-space (app-size: ${SIZE} KBytes, free: ${SIZE_FREE} KBytes)."
		exit 11
	fi

	VERSION_INSTALLED="${NAME} none"
else
	VERSION_INSTALLED=$(/usr/bin/expr match "$(/bin/cat /apps/${NAME}/VERSION)" ".*VERSION=\([0-9]\)")
	VERSION_INSTALLED="${NAME} ${VERSION_INSTALLED}"
fi

if [ "${no_start}" != "1" ]; then
	log "Stopping Cron-Daemon."
	/etc/init.d/S45cron stop &> /dev/null

	log "Stopping UserApp-Services."
	/etc/init.d/S91userapp stop &> /dev/null
fi

log "Remounting UserApp partition (rw)."
/bin/mount -oremount,rw ${USERAPP_HOME} || exit 12

if [ -d ${USERAPP_ROOT}/${NAME} ]; then
	log "Executing userapp pre_remove script."
	[ -x ${USERAPP_HOME}/setup/pre_remove.sh ] && ${USERAPP_HOME}/setup/pre_remove.sh ${NAME} update

	log "Cleaning up old Installation of '${NAME}'"
	rm -rf ${USERAPP_ROOT}/${NAME}
fi

log "Extracting new image (${pkg_file})."
cd ${USERAPP_ROOT} && tar zxf ${IMAGE_DIR}/${pkg_file} || exit 13

log "Executing post_install script."
[ -x ${USERAPP_HOME}/setup/post_install.sh ] && ${USERAPP_HOME}/setup/post_install.sh ${NAME}

log "Updating PATH and LD_LIBRARY_PATH variables in admin's .bash_profile."
for i in $(ls /apps/); do
	path_str_bin="${path_str_bin}:/apps/${i}/bin"
	path_str_lib="${path_str_lib}:/apps/${i}/lib"
done
path_str_bin=$(echo ${path_str_bin} | sed -e 's/\([[\/.*]\|\]\)/\\&/g')
path_str_lib=$(echo ${path_str_lib} | sed -e 's/\([[\/.*]\|\]\)/\\&/g')
sed "s/APPS_PATHS/${path_str_bin}/g" /home/admin/bash_profile_skel > /tmp/admin_bash_profile.tmp
sed "s/LIBS_PATHS/${path_str_lib}/g" /tmp/admin_bash_profile.tmp > /home/admin/.bash_profile
rm -f /tmp/admin_bash_profile.tmp

log "Setting Filesystem-Permissions."
chown -R admin:admin ${USERAPP_ROOT}/${NAME}

sync; sleep 1; sync; sleep 1

log "Remounting partition (ro)."
fstab_line=$(cat /etc/fstab | grep mtdblock7 | grep rw)
if [ -z "${fstab_line}" ]; then
	/bin/mount -oremount,ro ${USERAPP_HOME} || exit 14
else
	/bin/mount -oremount,rw ${USERAPP_HOME} || exit 14
fi

log "Cleaning up Temp-Directory."
tmp_cleanup

if [ "${no_start}" != "1" ]; then
	log "Restarting UserApp-Services."
	/etc/init.d/S91userapp start &> /dev/null

	log "Restarting Cron-Daemon."
	/etc/init.d/S45cron start &> /dev/null
fi

VERSION_TOINSTALL=$(expr match "$(cat /apps/${NAME}/VERSION)" ".*VERSION=\([0-9]*\)")
echo "[${TYPE}] $(date): ${VERSION_INSTALLED} => ${VERSION_TOINSTALL}" >> /usr/conf/update.log

exit 0
