#!/bin/bash
#
# This script is executed only once, immediately after the image
# tarball is unpacked to the app partition (/usr/local).
#

# global variables #
BEROCONF=/usr/fallback/beroconf
export PATH="/bin:/sbin:/usr/bin:/usr/sbin:/usr/fallback"

# functions #
function log {
	echo "[install_appfs_post] $(date): ${1}" | tee -a /usr/conf/permlog/install-appfs-post.log
}

function need_replace {

	old_c=$(grep "VERSION=[0-9]\{1,10\}" ${1})
	old_v=$(expr match "${old_c}" ".*=\([0-9]*\)")
	unset old_c

	new_c=$(grep "VERSION=[0-9]\{1,10\}" ${2})
	new_v=$(expr match "${new_c}" ".*=\([0-9]*\)")
	unset new_c

	if [ -z "${new_v}" ]; then
		echo 0
		return
	fi

	if [ -z "${old_v}" ] || [ ${old_v} -lt ${new_v} ]; then
		echo 1
		return
	fi

	echo 0
}

function block_installation {
	## not enough space on the mtdblock5.
	## to avoid huge issue, we block all installation by increasing
	## the PKG_VERSION from the VERSION.* files
	log ${1}
	/usr/fallback/beroconf set root rootfs-notenough-space ROOTFS_ERROR_NOTENOUGH_SPACE
	for file in $(ls /pkginfo/VERSION.*); do
		mount -o remount,rw /
		sed -i 's/PKG_VERSION=.*/PKG_VERSION=999/g' $file 
		mount -o remount,ro /
	done
	exit 0
}

# main #

if [ -f /usr/conf/permlog/install-appfs-post.log ]; then
	rm -f /usr/conf/permlog/install-appfs-post.log
fi

# check if this RootFS uses already the new Package-Mechanism
MIG2TO3=0
if [ ! -d "/pkginfo" ]; then
	MIG2TO3=1
fi

# cleaning rootfs space (if full)
available=$(df | grep /dev/mtdblock5 | awk '{ print $4; }')
if [ "${available}" -lt "250" ]; then
	log "Cleaning RootFS"
	## check if the pre-defined files list exist.
	if [ ! -f /usr/local/rootfs.files_list ]; then
		block_installation "rootfs.files_list missing"
	fi
	## cleaning unexpected files
	for file in $(find / -type f | grep -v /proc | grep -v /sys | grep -v /dev | grep -v /tmp | grep -v /usr/local | grep -v /usr/conf | grep -v /home/admin); do
		if cat /usr/local/rootfs.files_list | grep $file &>/dev/null ; then
			continue
		fi
		## avoid to remove /usr/bin/[
		if [ "$file" = "/usr/bin/[" ]; then
			continue
		fi
		mount -o remount,rw /
		rm $file
		mount -o remount,ro /
	done
	## update rootfs space size
	sync; sleep 1; sync;
	mount -o remount,rw /
	mount -o remount,ro /
	sync; sleep 1; sync;
	## check if enough space and cleaning worked
	available=$(df | grep /dev/mtdblock5 | awk '{ print $4; }')
	if [ "${available}" -lt "250" ]; then
		## still not enough space. we block the rootfs installation and display warning to customer
		block_installation "cleaning rootfs space failed"
	fi
	## force symlink for libcrypto and libssl
	mount -o remount,rw /
	ln -sf /home/admin/lib/libcrypto.so* /usr/lib
	mount -o remount,ro /
	mount -o remount,rw /
	ln -sf /home/admin/lib/libssl.so* /usr/lib
	mount -o remount,ro /
	sync; sleep 1; sync;
fi

# install rootfs-packages (we use the script supplied by this update)
log "Installing RootFS-Updates."
source /usr/local/conf/rootfs/VERSION.installer
if [ -z ${PKG_PATH} ];
then
	PGK_PATH=/usr/local
fi
${PKG_PATH}/conf/rootfs/pkg/installer/usr/sbin/rootfs_pkg_install.sh "ON_UPDATE"

# remove /etc/init.d/S50watchdog as it is part of AppFS
mount -oremount,rw /
if [ -f /etc/init.d/S50watchdog ]; then
	rm -f /etc/init.d/S50watchdog
fi
mount -oremount,ro /

# update conffs
log "Updating ConfFS."
conf_filelst="/usr/conf/snmpd_berofix.conf /usr/conf/isgw.tones /usr/conf/apiusers.conf"
for i in ${conf_filelst}; do
	conf_new="/usr/local/conf/conffs/${i//\/usr\/conf\//}"
	if [ "$(need_replace "${i}" "${conf_new}")" -eq 1 ] ; then
		cp ${conf_new} ${i}
	fi
done

# migrate isdn.conf to hardware.conf
if [ -f /usr/conf/isdn.conf ] ; then
	if grep -A2 "type=$" /usr/conf/isdn.conf | grep "clock=1" > /dev/null ; then
		sed "s/type=$/type=bf1E1/" /usr/conf/isdn.conf > /usr/conf/hardware.conf
		rm /usr/conf/isdn.conf
	else
		mv /usr/conf/isdn.conf /usr/conf/hardware.conf
	fi
	rm /usr/conf/ari.db
	/usr/bin/env -i bash -c "/usr/bin/php-cgi /usr/local/php/migration.php > /dev/null"
fi

# copy new SIP-profiles to /usr/conf
log "Installing SIP-Profiles."
if [ ! -d /usr/conf/profile ]; then
	mkdir -p /usr/conf/profile
fi
cp -a /usr/local/conf/conffs/profile/* /usr/conf/profile/

# create www-api-directory if it does not exist
if [ ! -d /usr/local/www/api ] ; then
    mkdir -p /usr/local/www/api
fi

# configure lan-mtu to sane value
if [ "$(${BEROCONF} get root lan-mtu)" = "500" ] ; then
        ${BEROCONF} set root lan-mtu 1500
fi

# apply changes to database and create files
log "Applying Changes to database."
/usr/bin/env -i bash -c "/usr/bin/php-cgi /usr/local/php/post_update.php  > /dev/null"
sync
sleep 1

insmod /lib/modules/berofix-driver.ko &> /dev/null
/usr/bin/env -i bash -c "/usr/bin/php-cgi /usr/local/php/create_files.php  > /dev/null"

# migrate beroconf database to replace "yes" by 1
if [ "$(${BEROCONF} get root lan-dhcp | grep -v failed)" = "yes" ]; then
	${BEROCONF} set root lan-dhcp 1
fi

# Reboot if RootFS has been transformed to Package-Mechanism
if [ ${MIG2TO3} -eq 1 ]; then
	sync; sleep 1; sync
	${BEROCONF} set root boot_fwupdate 0
	/bin/mount -oremount,ro /usr/local
	echo 1 > /tmp/reboot_after_update
#	/sbin/reboot
fi

